package de.Android_Seminar.backend.impl;

import de.Android_Seminar.backend.SampleRate;
import de.Android_Seminar.backend.SensorData;
import de.Android_Seminar.backend.SensorType;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.location.Location;
import android.util.Log;

public abstract class SensorDataRecorder implements SensorEventListener {
	
	/** the amount of data sets the a buffers should contain **/
	protected int DATA_BUFFER_SIZE = 60 ;
	
	/** the number of values the sensor delivers **/
	protected int SENSOR_VALUE_NUMBER;
	
	/** a buffer where the corresponding sensor stores it's data to **/
	private DataBuffer buffer;
	
	/** the type of this sensor **/
	protected SensorType sensorType;
	
	/** the sample rate, with which the sensor will record **/
	protected SampleRate sampleRate;
	
	/** the latest recognized location **/
	private Location location;	
	
	/**
	 * 
	 * @param manager 
	 * @param sensorType
	 * @param sampleRate
	 */
	public SensorDataRecorder(SensorManager manager, SampleRate sampleRate) {
		initSensorParameters();
		
		buffer = new DataBuffer(DATA_BUFFER_SIZE, SENSOR_VALUE_NUMBER, this.sensorType);
		tmpData = new SensorDataImpl(sensorType, SENSOR_VALUE_NUMBER);
		tmpArray = new double[SENSOR_VALUE_NUMBER];
		this.sampleRate = sampleRate;
		activateSensor(manager);		
	}
	
	/**
	 * 
	 * @param manager 
	 * @param sensorType
	 * @param sampleRate
	 */
	public SensorDataRecorder(SensorManager manager, SampleRate sampleRate, int bufferSize) {
		
		initSensorParameters();
		DATA_BUFFER_SIZE = bufferSize;
		
		buffer = new DataBuffer(DATA_BUFFER_SIZE, SENSOR_VALUE_NUMBER, this.sensorType);
		tmpData = new SensorDataImpl(sensorType, SENSOR_VALUE_NUMBER);
		tmpArray = new double[SENSOR_VALUE_NUMBER];
		this.sampleRate = sampleRate;
		activateSensor(manager);		
	}
	
	/**
	 * initializes some parameters of the sensor.
	 * This method should initialize {@link SensorDataRecorder#SENSOR_VALUE_NUMBER} and {@link SensorDataRecorder#sensorType}
	 * and optionally {@link SensorDataRecorder#DATA_BUFFER_SIZE}
	 * This method MUST be the first method this class ({@link SensorDataRecorder}) calls
	 */
	protected abstract void initSensorParameters();
	
	/**
	 * 
	 * @return the 'general' max value a sensor expected to deliver
	 */
	protected abstract double getMaxValue();
	
	/**
	 * 
	 * @return the 'general' min value a sensor is expected to deliver
	 */
	protected abstract double getMinValue();
	
	public void onAccuracyChanged(Sensor sensor, int accuracy) {
		// not needed		
	}

	private SensorData tmpData;
	private double[] tmpArray;
	public void onSensorChanged(SensorEvent event) {
		
		tmpArray = new double[SENSOR_VALUE_NUMBER];	
		for(int i=0; i<SENSOR_VALUE_NUMBER; i++){
			tmpArray[i] = event.values[i];
		}
		
		
		long time =  System.currentTimeMillis();
		tmpData.setValues(tmpArray,time);
		tmpData.setLocation(location);
		
		buffer.addData(tmpData);
	}
	
	/**
	 * @return the latest data set the sensor has delivered
	 * @see SensorData
	 */
	public SensorData getCurrentData() {
	
		return buffer.getData();
	}
	
	/**
	 * @return all the data the sensor's corresponding data buffer contains 
	 */
	public SensorData[] getBufferedData(int amount) {
		
		return buffer.getData(amount);
	}
	
	/**
	 * sets up a sensor
	 * @param manager the SensorManager where a sensor can be obtained
	 * @param sampleRate the sample rate with which the sensor should record data
	 * @see SensorManager
	 * @see SampleRate
	 */
	public void activateSensor(SensorManager manager) {
		
		/* initialize sensor */
		int lSensorType = SensorMapping.getAndroidSensor(sensorType);
		int lSampleRate = 0;
		switch (sampleRate) {
		case FASTEST:
			lSampleRate = SensorManager.SENSOR_DELAY_FASTEST;
			break;
		case FAST:
			lSampleRate = SensorManager.SENSOR_DELAY_GAME;
			break;
		case NORMAL:
			lSampleRate = SensorManager.SENSOR_DELAY_NORMAL;
			break;
		case SLOW:
			lSampleRate = SensorManager.SENSOR_DELAY_UI;
			break;
		}
		if(manager.getSensorList(lSensorType).size() > 0) {
			Sensor sensor = manager.getSensorList(lSensorType).get(0);
			manager.registerListener(this, sensor, lSampleRate );
			
			//del:
//			Log.d("status", "RANGE: " +sensor.getMaximumRange());
		}
	}
	
	/**
	 * This method unregisters a sensor from the android system
	 * @param manager {@link SensorManager}
	 */
	public void deactivateSensor(SensorManager manager) {
		
		int lSensorType = SensorMapping.getAndroidSensor(sensorType);
		Sensor sensor = manager.getSensorList(lSensorType).get(0);
		manager.unregisterListener(this, sensor);
	}
	
	/**
	 * updates the current location, thus the sensor can create the correct {@link SensorData}
	 * @param location the new location
	 */
	public void updateLocation(Location location) {
		
		this.location = location;
	}

}
