package de.Android_Seminar.backend.impl;

import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import android.content.res.XmlResourceParser;
import android.util.Log;
import android.util.Xml;
import de.Android_Seminar.backend.SampleRate;
import de.Android_Seminar.backend.SensorConfiguration;
import de.Android_Seminar.backend.SensorController;
import de.Android_Seminar.backend.SensorType;

public class SensorConfigurationImpl implements SensorConfiguration {

	/** the xml file, where all the parameters will be saved to and read from **/	
	private static final String xmlFile = "sdcard/loggerConfig.xml";
	
	/** the default sample rate for all the sensors **/
	private static final SampleRate DEFAULT_SAMPLE_RATE = SampleRate.NORMAL;
	
	/** the default start delay (in seconds) for a recording **/
	private static final float LAST_START_DELAY = 1.0f;
	
	/** the default time, after a recording stops. -1 means infinitely time**/
	private static final float DEFAULT_TIMEOUT = -1f;
	
	/** default value for lastLocationSaving **/
	private static final boolean DEFAULT_LOCATION_SAVING = false;
	
	/** default value for lastTimeSaving **/
	private static final boolean DEFAULT_TIME_SAVING = false;
	
	/** the default sensors which should be used for a recording **/
	private List<SensorType> LAST_USED_SENSORS = new ArrayList<SensorType>();
	
	/** a collection that contains sample rates, that where last use, for the corresponding sensors **/
	private Map<SensorType, SampleRate> lastSampleRates;
	
	/** the time, after a recording stops. -1 means infinitely time**/
	private float lastTimeout;
	
	/** the latest used start delay for a recording. 
	 *  @see SensorController#startRecording(String tag, String file, float startDelay) 
	 *  **/
	private float lastStartDelay;
	
	/** all sensors that have been used for the last recording **/
	private Set<SensorType> lastUsedSensors;
	
	/** determines if the location of an activity was saved last time **/ 
	private boolean lastLocationSaving;
	
	/** determines if the time stamp of an activity was saved last time **/ 
	private boolean lastTimeSaving;
	
	/**
	 * standard constructor
	 */
	public SensorConfigurationImpl() {
		
		Log.d("status", "create SensorConfigurationImpl");
		
		// initialization
		LAST_USED_SENSORS.add(SensorType.ACCELEROMETER);
		lastSampleRates = new HashMap<SensorType, SampleRate>();
		lastUsedSensors = new TreeSet<SensorType>();
			
		parseXmlFile();
	}
	
	public Map<SensorType, SampleRate> getLastSampleRates() {

		return lastSampleRates;
	}

	public float getLastStartDelay() {

		return lastStartDelay;
	}

	public Set<SensorType> getLastUsedSensors() {
		
		return lastUsedSensors;
	}

	public void setLastSampleRates(Map<SensorType, SampleRate> rates) {
		
		lastSampleRates.clear();
		lastSampleRates = rates;
		
		saveInXML();
	}

	public void setLastStartDelay(float delay) {
		
		this.lastStartDelay = delay;
		
		saveInXML();
	}

	public void setLastUsedSensors(Set<SensorType> sensors) {
		
		lastUsedSensors.clear();
		lastUsedSensors = sensors;
		
		saveInXML();
	}
	
	/**
	 * this method parses a xml file
	 * @param file the file which sould be parsed
	 */
	private void parseXmlFile() {
		
		
		
		URL url;
		try {
			url = new URL("file:///" +xmlFile);
		
		
		XmlPullParser parser = Xml.newPullParser();
        
		try {
			parser.setInput(url.openStream(), null);
			
			
			int event_type = parser.getEventType();
			boolean done = false;
			String name = null;
			
			// parser specific:
			boolean lastUsedSensorsActive = false;
			boolean lastUsedSampleRatesActive = false;
			
			while (event_type != XmlResourceParser.END_DOCUMENT && !done){
				
				name = parser.getName();
				
				switch (event_type){
					case XmlResourceParser.START_DOCUMENT:
						
						break;
						
					case XmlResourceParser.START_TAG:

						if(name.equalsIgnoreCase("lastUsedSensors"))
							lastUsedSensorsActive = true;
						
						
						if(name.equalsIgnoreCase("lastUsedSampleRates"))
							lastUsedSampleRatesActive = true;
						
						
						if(name.equalsIgnoreCase("startUpDelay")) {
							String delayString = parser.getAttributeValue(null, "delay");
							this.lastStartDelay = Float.parseFloat(delayString);
						}
						
						if(name.equalsIgnoreCase("timeout")) {
							String timeoutString = parser.getAttributeValue(null, "value");
							this.lastTimeout = Float.parseFloat(timeoutString);
						}
						
						if(name.equalsIgnoreCase("timesaving")) {
							String value = parser.getAttributeValue(null, "value");
							this.lastTimeSaving = Boolean.parseBoolean(value);
						}
						
						if(name.equalsIgnoreCase("locationsaving")) {
							String value = parser.getAttributeValue(null, "value");
							this.lastLocationSaving = Boolean.parseBoolean(value);
						}
						
						if(name.equalsIgnoreCase("sensor")) {
							if(lastUsedSensorsActive) {
								String sensorString = parser.getAttributeValue(null, "type");
								this.lastUsedSensors.add(SensorType.valueOf(sensorString));
							}
							
							if(lastUsedSampleRatesActive) {
								String sensorString = parser.getAttributeValue(null, "type");
								String rate = parser.getAttributeValue(null, "rate");
								this.lastSampleRates.put(SensorType.valueOf(sensorString), SampleRate.valueOf(rate));
							}
						}
						
						break;
						
					case XmlResourceParser.END_TAG:
						
						if(name.equalsIgnoreCase("lastUsedSensors"))
							lastUsedSensorsActive = false;
								
						if(name.equalsIgnoreCase("lastUsedSampleRates"))
							lastUsedSampleRatesActive = false;
						
						break;
				}
				
				event_type = parser.next();
			}
			
		} catch (XmlPullParserException e) {
			e.printStackTrace();			
		} catch (IOException e) {
			createDefaultXml();
		}
		} catch (MalformedURLException e1) {
			e1.printStackTrace();
		}
	}
	
	/**
	 * saves all parameters to the configuration file
	 */
	private void saveInXML() {

			try {
				FileWriter fw = null;
				fw = new FileWriter( "sdcard/loggerConfig.xml" , false );
				PrintWriter pw = new PrintWriter( fw );

				pw.println("<?xml version='1.0' encoding='UTF-8' standalone='yes' ?>");
				
				pw.println("<sensorConfig>");
				
				pw.println("<lastUsedSensors>");
				Iterator<SensorType> sensIter = lastUsedSensors.iterator();
				while(sensIter.hasNext()) {
					pw.println("<sensor type=\"" +sensIter.next().name() +"\"/>");
				}
				pw.println("</lastUsedSensors>");
				
				
				pw.println("<lastUsedSampleRates>");
		        Set<SensorType> sensors = lastSampleRates.keySet();
		        Iterator<SensorType> iter = sensors.iterator();
				while(iter.hasNext()) {
		        	SensorType s = iter.next();
		        	pw.println("<sensor type=\"" +s.name() +"\" rate=\"" +lastSampleRates.get(s).name() +"\"/>");
		        }
				pw.println("</lastUsedSampleRates>");
				
				
				pw.println("<startUpDelay delay=\"" +lastStartDelay +"\"/>");
				
				pw.println("<timeout value=\"" +lastTimeout +"\"/>");
				
				pw.println("<timesaving value=\"" +lastTimeSaving +"\"/>");
				
				pw.println("<locationsaving value=\"" +lastLocationSaving +"\"/>");
				
				pw.println("</sensorConfig>");

				fw.flush();
				fw.close();
				pw.flush();
				pw.close();
			} catch (IOException e) {
				Log.d("error", e.getMessage());
				e.printStackTrace();
			}
	}
	
	/**
	 * creates a xml file to the location: xmlFile and uses the default values 
	 */
	private void createDefaultXml() {
		
		lastSampleRates.clear();
		lastSampleRates.put(LAST_USED_SENSORS.get(0), DEFAULT_SAMPLE_RATE);
		
		lastUsedSensors.clear();
		lastUsedSensors.add(LAST_USED_SENSORS.get(0));
		
		lastStartDelay = LAST_START_DELAY;
		this.lastTimeout = DEFAULT_TIMEOUT;
		this.lastLocationSaving = DEFAULT_LOCATION_SAVING;
		this.lastTimeSaving = DEFAULT_TIME_SAVING;
		
		saveInXML();
	}

	public float getLastTimeout() {

		return lastTimeout;
	}

	public void setLastTimeout(float timeout) {
		
		this.lastTimeout = timeout;
	}

	public boolean getLocationSaving() {

		return lastLocationSaving;
	}

	public boolean getTimeSaving() {

		return lastTimeSaving;
	}

	public void setLocationSaving(boolean saving) {

		lastLocationSaving = saving;
	}

	public void setTimeSaving(boolean saving) {

		lastTimeSaving = saving;
	}

}
