package de.Android_Seminar.backend.impl;

import android.util.Log;
import de.Android_Seminar.backend.SensorData;
import de.Android_Seminar.backend.SensorType;

/**
 * A Buffer that stores data, delivered from a sensor.
 *   
 * @author jonas
 *
 */
public class DataBuffer {
	
	
	/** the number of data sets a buffers should be able to hold **/
	private final int DATA_SET_NUMBER;
		
	/** the buffer where the data have to be saved **/
	private SensorData[] dataBuffer;
		
	/** marks the position where to write data to **/
	private int writePointer;
	
	/** amount of data sets already stored in the buffer**/
	private int count;
	
	/** the sensor type of the sensor that stores values in this buffer **/
	private SensorType sensorType;
	
	/**
	 * 
	 * @param dataSetNumber the number of data sets a buffers should be able to hold
	 * @param dataSetLength the number of values one data set holds.
	 * @param sensor the sensor type of the sensor that stores values in this buffer
	 * 
	 * @see SensorData
	 * @see SensorType
	 */
	public DataBuffer(int dataSetNumber, int dataSetLength, SensorType sensor) {
		
		sensorType = sensor;
		DATA_SET_NUMBER = dataSetNumber;
		dataBuffer = new SensorData[dataSetNumber];
		for(int i=0; i<DATA_SET_NUMBER; i++)
			dataBuffer[i] = new SensorDataImpl(sensor, dataSetLength);
		
		writePointer = -1;
		count = 0;
		
		tValues = new double[dataSetLength];
	}
	
	/**
	 * adds a new data set to the buffer. if the buffer is full,
	 * the oldest data set will be overwritten
	 * @param data the new data set that will be added to the buffer
	 */
	double[] tValues;
	public void addData(SensorData data) {
		
		writePointer++;
		if(writePointer >= DATA_SET_NUMBER)
			writePointer = 0;
		
		double[] tempValues = data.getValues();
		
		for(int i=0; i<tempValues.length; i++)
			tValues[i] = tempValues[i];	
			
		dataBuffer[writePointer].setValues(tValues, data.getTimeStamp().getTime());
		dataBuffer[writePointer].setLocation(data.getLocation());
		
		if(count <= DATA_SET_NUMBER)
			count++;
	}	
	
	/**
	 * @return the newest data set from the buffer or null if the buffer is empty
	 */
	public SensorData getData() {
		if(writePointer < 0)
			return dataBuffer[0];
		
		return dataBuffer[writePointer];
	}
	
	/**
	 * @return all the data, contained in the buffer
	 */
	synchronized public SensorData[] getData(int amount) {
		
		if(writePointer < 0){
//			SensorData[] s = new SensorData[0];
////			Log.d("status", "data: " +(s==null));
			writePointer = 0;
//			return s;
		}
				
		if(amount > count) {
			if(count > DATA_SET_NUMBER) {
				amount = DATA_SET_NUMBER;
			}
			else {
				if(count > 0) {
					amount = count;
				}
			}
		}
				
		SensorData[] returnArray = new SensorData[amount];
		
		for(int i=writePointer; i<amount; i++) {
			returnArray[i] = dataBuffer[i];
		}
		for(int i=0; i<writePointer; i++)
			returnArray[i] = dataBuffer[i];
		

		return returnArray;
	}
	
	/**
	 * @return the sensor type for which data can be saved in this buffer
	 * @see SensorType
	 */
	public SensorType getSensorType() {
		
		return sensorType;
	}	
	
	/**
	 * clears the buffer
	 */
	public void clear() {
		
		count = 0;
		writePointer = -1;
	}
}
