package de.Android_Seminar.backend;

import java.sql.Timestamp;

import android.location.Location;

/**
 * A container interface for sensor data.
 * The sensor data from the different sensors varies sometimes, thus 
 * some conventions are needed.
 * For example the sensor data for a accelerometer consists of 3 values,
 * values for the x-, y-, and z-axis of the accelerometer. The temperature
 * sensor on the other hand only delivers one value.
 * Thus a data set consists of all these values, delivered by a sensor
 * as well as its corresponding sensor and the time these data have been recorded. 
 * 
 * @author jonas
 *
 */
public interface SensorData {

	/**
	 * @return all values for one data set
	 * e.g. a accelerometer data set consists of 3 values (x, y, z)
	 * a temperature data set consists of only 1 value
	 * @see SensorData#setValues(float[])
	 */
	public double[]  getValues();
	
	/**
	 * @param values sets the values for one data set
	 * e.g. a accelerometer data set consists of 3 values (x, y, z)
	 * a temperature data set consists of only 1 value
	 * @param timestamp the time when the data was changed 
	 * (time in milliseconds since January 1, 1970 00:00:00 UTC)
	 * @see SensorData#getValues()
	 */
	public void setValues(double[] values, long timestamp);
	
	/**
	 * @return the number of values, a data set consists of.
	 * e.g. a accelerometer data set consists of 3 values (x, y, z)
	 * a temperature data set consists of only 1 value
	 */
	public int getLength();
	
	/**
	 * @return the sensor type for which data can be saved in this data set
	 * @see SensorType
	 */
	public SensorType getSensorType();
	
	/**
	 * @return the time stamp this a data set has been recorded
	 * @see Timestamp
	 */
	public Timestamp getTimeStamp();
	
	/**
	 * @return the location at which the data set has been recorded
	 * @see Location
	 */
	public Location getLocation();
	
	/**
	 * @param location the location at which the data set will be recorded
	 */
	public void setLocation(Location location);
	
	/**
	 * 
	 * @param timeStamp sets the time stamp when a data set was recorded
	 * @see Timestamp
	 * @deprecated do not use this method any more
	 */
	public void setTimeStamp(Timestamp timeStamp);
}
