package de.Android_Seminar.backend;

import java.util.List;

import de.Android_Seminar.fileTransmitter.FileTransmitter;
import de.Android_Seminar.xmlCreator.XmlCreator;

/**
 * An interface to control the devices sensors, start a recording for
 * different sensors and obtain data from them.
 * 
 * @author jonas
 *
 */
public interface SensorController {

	/**
	 * Starts a data recording. Data will be collected from all 'active' sensors.
	 * A recording will continue until the {@link SensorConfiguration#getLastTimeout()} value is reached.
	 * A recording will start after {@link SensorConfiguration#getLastStartDelay()} seconds.
	 * 
	 * @param tag determines how a data set will be tagged. If the tag is a {@code null} value
	 * @param file the path and name of the xml file (audio file) the data will be saved to.
	 * If audio data will be recorded, for this data type a audio file (mp3) will be created.
	 * Thus all data, except audio data will be saved in a single xml file with path and name
	 * specified in the 'file' attribute. For audio data, a mp3 file will be created with path 
	 * and name specified in the 'file' attribute.
	 * If no tag should be set, an empty string or a null value can be delivered.
	 * @see SensorController#stopRecording()
	 * @see SensorController#activateSensor(SensorType)
	 * @see SensorController#deactivateSensor(SensorType)
	 */
	public void startRecording(String tag, String file);			
				
	//TODO: listener
	
	/**
	 * Stops a recording immediately. Sensors will stay 'active' but no data will
	 * be recorded.
	 * 
	 * @see SensorController#startRecording(String, String, float)
	 * @see SensorController#activateSensor(SensorType)
	 * @see SensorController#deactivateSensor(SensorType)
	 */
	public void stopRecording();
	
	/**
	 * Determines if a recording is running
	 * 
	 * @return true if data is recorded, false otherwise
	 * @see SensorController#startRecording(String, String, float)
	 * @see SensorController#stopRecording()
	 */
	
	public boolean isRecordin();
	
	/** 
	 * Activates a sensor and makes it ready to record data.
	 * Before a sensor is activated, its sample rate should be set up
	 * in the {@link SensorConfiguration} corresponding to this {@link SensorController} 
	 * object. Otherwise the default sample rate determined by the {@link SensorConfiguration} 
	 * class will be used.
	 * 
	 * @param sensor the sensor which should be activated 
	 * @see SensorController#startRecording(String, String, float)
	 * @see SensorController#deactivateSensor(SensorType)
	 */
	public void activateSensor(SensorType sensor);
	
	/**
	 * 
	 * @param sensor the sensor which should be activated
	 * @param bufferSize the size of the sensor's buffer. If the size is smaller than
	 * one, the default size will be used
	 * @see SensorController#activateSensor(SensorType)
	 */
	public void activateSensor(SensorType sensor, int bufferSize);
	
	/** 
	 * Disables a sensor and prevents it from recording data.
	 * 
	 * @see SensorController#startRecording(String, String, float)
	 * @see SensorController#activateSensor(SensorType)
	 * @see SensorType
	 */
	public void deactivateSensor(SensorType sensor);
	
	/**
	 * 
	 * @return a list of all sensors that are active.
	 * @see SensorController#activateSensor(SensorType)
	 * @see SensorType
	 */
	public List<SensorType> getActiveSensors();
	
	/**
	 * @return number of sensors that are active.
	 * @see SensorController#activateSensor(SensorType)
	 * @see SensorType 
	 */
	public int getDeviceSensorNumber();
	
	/**
	 * @return a list of all available sensors of the device
	 * @see SensorType
	 */
	public List<SensorType> getDeviceSensors();
	
	/**
	 * @return a SensorConfiguration for this {@link SensorController}
	 * @see SensorConfiguration
	 */
	public SensorConfiguration getSensorConfiguration();
	
	/**
	 * Determines if the device has a specific sensor
	 * @param sensor the sensor which should be checked for availability
	 * @return true, if the device has such a sensor, false otherwise.
	 * @see SensorType
	 */
	public boolean hasSensor(SensorType sensor);
	
	/**
	 * Checks, if a sensor is active or not
	 * 
	 * @param sensor the sensor which should be checked
	 * @return true, if the sensor is active, false otherwise
	 * @see SensorType
	 */
	public boolean sensorIsActive(SensorType sensor);
	
	/**
	 * returns an sensor specific, expected maximum value. The sensor
	 * has to be activated.
	 * @param sensor the sensor where the maximal value should be obtained from
	 * @return the maximal value the sensor can deliver
	 * @see SensorType
	 */
	public double getMaxValue(SensorType sensor); 
	
	/**
	 * returns an sensor specific, expected minimum value. The sensor
	 * has to be activated.
	 * @param sensor the sensor where the minimal value should be obtained from
	 * @return the minimal value the sensor can deliver
	 * @see SensorType
	 */
	public double getMinValue(SensorType sensor); 
	
	/**
	 * @param sensor the sensor where the data should be obtained from
	 * @return the latest data set the sensor has recorded; null if no sensor
	 * of type {@link SensorType} was activated ({@link SensorController#activateSensor(SensorType)}),
	 * or no data have been received already.
	 * @see SensorType
	 * @see SensorData
	 */
	public SensorData getCurrentData(SensorType sensor);
	
	/**
	 * @param sensor the sensor where the data should be obtained from
	 * @param amount the number of data sets that should be returned.
	 * If 'amount' is greater than the sernsor's buffer size, all buffered
	 * data will be returned.
	 * @return an array with latest 'amount' sensor data; null if no sensor
	 * of type {@link SensorType} was activated: ({@link SensorController#activateSensor(SensorType)}).
	 * An empty array if no data were received from the sensor
	 * @see SensorType
	 * @see SensorData
	 */
	public SensorData[] getBufferedData(SensorType sensor, int amount);
		
	/**
	 * closes the SensorController and disables all sensors.
	 * 
	 *  @see SensorController#deactivateSensor(SensorType)
	 */
	public void close();
	
    /**
     * 
     * @return a {@link XmlCreator} that gives you some xml creating/manipulation features
     * for activities
     */
	public XmlCreator getXmlCreator();
	
	/**
	 * 
	 * @return a {@ FileTransmitter} that supplies features for transmitting a file to a server
	 */
	public FileTransmitter getFileTransmitter();
}
