package de.Android_Seminar.backend;


import java.util.Map;
import java.util.Set;

/**
 * An interface to set up some parameter for the different sensors.
 * The parameters will be saved in a configuration file on the sd card.
 * If there is no such file on the sd card, a new file with default values
 * will be created.
 * This interface can be used to prefill some fields in the GUI with the last values
 * that where used.
 *  
 * @author jonas
 *
 */
public interface SensorConfiguration {
	
	
	/**
	 * Receives a list of all sensors that have been used during the last
	 * recording. If the application just started, the list will be obtained
	 * from the configuration file. 
	 * 
	 * @return the sensors that have been used during the last recording.
	 * @see SensorType
	 */
	public Set<SensorType> getLastUsedSensors();
	
	/**
	 * Saves all sensors that have been used for the last recording to the configuration file.
	 * 
	 * @param sensors all sensors that have been used for the last recording
	 * @see SensorType
	 */
	public void setLastUsedSensors(Set<SensorType> sensors);
	
	/**
	 * Receives the last start delay. If the application just started, the
	 * start delay will be obtained from the configuration file.
	 * 
	 * @return the last start delay time.
	 * @see SensorController#startRecording(String, String, float)
	 */
	public float getLastStartDelay();
	
	/**
	 * Saves the last start delay to the configuration file.
	 * 
	 * @param delay the time the application has to wait until the
	 * recording will start after executing the {@link SensorController#startRecording(String, String, float)}
	 * method.
	 * @see SensorController#startRecording(String, String, float)
	 */
	public void setLastStartDelay(float delay);
	
	/**
	 * Receives the last for a specified sensor. If the application just started, the
	 * last sample rate will be obtained from the configuration file. 
	 * 
	 * @param sensor the sensor for which the last sample rate should be returned
	 * @return the last sample rate used for specified sensor
	 * @see SensorType
	 */
	public Map<SensorType, SampleRate> getLastSampleRates();
	
	/**
	 * Saves the last sample rate to the configuration file.
	 * 
	 * @param rates the sample rate and the sensor for which the last sample rate should be set
	 * @see SampleRate
	 * @see SensorType
	 */
	public void setLastSampleRates(Map<SensorType, SampleRate> rates);
	
	/**
	 * @return the last timeout value (in seconds). This value determines when a data recording will stop.
	 * if this value is negative, the recording will never stop.
	 */
	public float getLastTimeout();
	
	/**
	 * The timeout value determines when a data recording will stop.
	 * if this value is negative, the recording will never stop.
	 * @param timeout the timeout in seconds
	 */
	public void setLastTimeout(float timeout);
	
	/**
	 * determines whether the current location should be saved within an 
	 * 'activity' in the xml file or not 
	 * @param saving if this is true, the location will be saved
	 */
	public void setLocationSaving(boolean saving);
	
	/**
	 * @return true, if the location will be saved, false otherwise.
	 * @see SensorController#setLocationSaving(boolean)
	 */
	public boolean getLocationSaving();
	
	/**
	 * determines whether the current time should be saved within an 
	 * 'activity' in the xml file or not 
	 * @param saving if this is true, the time will be saved
	 */
	public void setTimeSaving(boolean saving);
	
	/**
	 * @return true, if the time will be saved, false otherwise.
	 * @see SensorController#setTimeSaving(boolean)
	 */
	public boolean getTimeSaving();
		
}
